/* irt.c
 *
 * Copyright (C) 2005 Stephane Germain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
   \file

   \brief A command line program to estimate items and abilities
   in a IRT model.

   \author Stephane Germain <germste@gmail.com>
*/

#include <libirt.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <getopt.h>

/**
   \brief The command line options.

   To be used by getopt_long.
*/
struct option long_options[] = {
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, 'v'},
  {"nbr_item", required_argument, 0, 'i'},
  {"nbr_subject", required_argument, 0, 'j'},
  {"skip", required_argument, 0, 'k'},
  {"offset", required_argument, 0, 'o'},
  {"extra", required_argument, 0, 'x'},
  {"delimiter", required_argument, 0, 'd'},
  {"nbr_quad", required_argument, 0, 'q'},
  {"model", required_argument, 0, 'm'},
  {"penalized", no_argument, 0, 'P'},
  {"kernel", no_argument, 0, 'K'},
  {"smooth_factor", required_argument, 0, 's'},
  {"no_priors", no_argument, 0, 'N'},
  {"slope_mean", required_argument, 0, 1},
  {"slope_dev", required_argument, 0, 2},
  {"thresh_mean", required_argument, 0, 3},
  {"thresh_dev", required_argument, 0, 4},
  {"asymp_mean", required_argument, 0, 5},
  {"asymp_weight", required_argument, 0, 6},
  {"no_slope_prior", no_argument, 0, '7'},
  {"thresh_prior", no_argument, 0, 8},
  {"no_asymp_prior", no_argument, 0, 9},
  {"max_em_iter", required_argument, 0, 'e'},
  {"max_nr_iter", required_argument, 0, 'n'},
  {"response_functions", required_argument, 0, 'R'},
  {"grouping", no_argument, 0, 'G'},
  {"verbose", no_argument, 0, 'V'},
  {"ability", no_argument, 0, 'A'},
  {"no_ability_prior", no_argument, 0, 10},
  {"quad_from", required_argument, 0, 11},
  {"quad_to", required_argument, 0, 12},
  {"true", required_argument, 0, 13},
  {"false", required_argument, 0, 14},
  {"blank", required_argument, 0, 15},
  {"trimming", no_argument, 0, 'T'},
  {"slope_init", required_argument, 0, 16},
  {"thresh_init", required_argument, 0, 17},
  {"asymp_init", required_argument, 0, 18},
  {"precision", required_argument, 0, 19},
  {0, 0, 0, 0}
};

/** \brief The file name. */
char *file_name;

/** \brief The number of item. */
int nbr_item;

/** \brief The number of subject. */
int nbr_subject;

/** \brief The number of line to skip. */
int skip;

/** \brief The number of character or field to skip. */
int offset;

/** \brief The number of character or field between items. */
int extra;

/** \brief Enable the use of delimited format file. */
int delimited;

/** \brief The delimiter. */
char delimiter;

/** \brief The string representing a success. */
char *true_string;

/** \brief The string representing a failure. */
char *false_string;

/** \brief The string representing a non response. */
char *blank_string;

/** \brief The number of quadrature class. */
int nbr_quad;

/** \brief The first class middle point. */
double quad_from;

/** \brief The last class middle point. */
double quad_to;

/** \brief The model to use. */
int model;

/** \brief Enable the use of a prior on the slope. */
int slope_prior;

/** \brief Enable the use of a prior on the threshold. */
int thresh_prior;

/** \brief Enable the use of a prior on the asymptote. */
int asymp_prior;

/** \brief The mean of the prior on the slope. */
double slope_mean;

/** \brief The standard deviation of the prior on the slope. */
double slope_dev;

/** \brief The mean of the prior on the threshold. */
double thresh_mean;

/** \brief The standard deviation of the prior on the threshold. */
double thresh_dev;

/** \brief The mean of the prior on the asymptote. */
double asymp_mean;

/** \brief The weight of the prior on the asymptote. */
double asymp_weight;

/** \brief The initial value of the slope. */
double slope_init;

/** \brief The initial value of the threshold. */
double thresh_init;

/** \brief The initial value of the asymptote. */
double asymp_init;

/** \brief The maximum number of EM iteration. */
int max_em_iter;

/** \brief The maximum number of newton iteration. */
int max_nr_iter;

/** \brief The precision. */
double precision;

/** \brief Enable the grouping of identical patterns. */
int grouping;

/** \brief The verbosity level. */
int verbose;

/** \brief Enable the estimation of the abilities. */
int ability;

/** \brief Enable the use of EAP to estimate the abilities. */
int ability_prior;

/** \brief Enable the trimming of spaces in the fields (when delimited). */
int trimming;

/** \brief Enable the use of penalized maximum marginal likelihood. */
int penalized;

/** \brief Enable the use of kernel estimators. */
int kernel;

/** \brief The smooth factor if PMMLE or kernel is used. */
double smooth_factor;

/** \brief Enable the display of the response functions instead of the parameters. */
int response_functions;

/**
   \brief Parse the options and arguments of the program.

   Also set the defaults values.

   \return 1 for success and 0 for failure.
*/
int
parse_arg (int argc, char *argv[])
{
  int opt = 0;
  int option_index = 0;

  /* the defaults */
  file_name = NULL;
  nbr_item = 0;
  nbr_subject = 0;
  skip = 0;
  offset = 0;
  extra = 0;
  delimited = 1;
  delimiter = ',';
  true_string = "1";
  false_string = "0";
  blank_string = NULL;
  nbr_quad = 64;
  quad_from = -4.0;
  quad_to = 4.0;
  model = 2;
  slope_prior = 1;
  thresh_prior = 0;
  asymp_prior = 1;
  slope_mean = 1.702;
  slope_dev = 1.649*1.702;
  thresh_mean = 0.0;
  thresh_dev = 2.0;
  asymp_mean = 0.2;
  asymp_weight = 20.0;
  slope_init = slope_mean;
  thresh_init = thresh_mean;
  asymp_init = asymp_mean;
  max_em_iter = 100;
  max_nr_iter = 100;
  precision = 1e-5;
  grouping = 0;
  verbose = 0;
  ability = 0;
  ability_prior = 1;
  trimming = 0;
  penalized = 0;
  kernel = 0;
  smooth_factor = 0.0;
  response_functions = 0;

  opterr = 1;

  /* the parsing itself */
  while ((opt =
	  getopt_long (argc, argv, "hi:j:k:o:x:d:q:m:PKe:n:RGVANWTvs:",
		       long_options, &option_index)) != -1)
    {
      switch (opt)
	{
	case 0:
	  break;
	case 'h':
	  printf ("USAGE : irt [OPTION] file_name\n");
	  printf ("\n");
	  printf
	    ("Estimate by MMLE (marginal maximum likelihood) the parameters\n");
	  printf
	    ("of each item in an binary IRT (item response theory) model.\n");
	  printf ("\n");
	  printf
	    ("Each line of the input file should contains the responses of a\n");
	  printf
	    ("subject to each item. By default the responses are assumed to\n");
	  printf ("be comma delimited with 0 for failure, 1 for success,\n");
	  printf ("and anything else for non-response.\n");
	  printf ("\n");
	  printf ("Consult the manual fore more information.\n");
	  printf ("\n");
	  printf ("Report bugs to <germste@gmail.com>.\n");
	  exit (0);
	  break;
	case 'v':
	  printf ("Using version %s of libirt.\n", libirt_version);
	  printf ("Copyright (C) 2005 Stephane Germain.\n");
	  printf ("libirt comes with NO WARRANTY,\n");
	  printf ("to the extent permitted by law.\n");
	  printf ("You may redistribute copies of libirt\n");
	  printf ("under the terms of the GNU General Public License.\n");
	  printf ("For more information about these matters,\n");
	  printf ("see the file named COPYING.\n");
	  exit (0);
	  break;
	case 'i':
	  nbr_item = atoi (optarg);
	  if (nbr_item < 0)
	    {
	      printf ("nbr_item should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'j':
	  nbr_subject = atoi (optarg);
	  if (nbr_subject < 0)
	    {
	      printf ("nbr_subject should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'k':
	  skip = atoi (optarg);
	  if (skip < 0)
	    {
	      printf ("skip should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'o':
	  offset = atoi (optarg);
	  if (offset < 0)
	    {
	      printf ("offset should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'x':
	  extra = atoi (optarg);
	  if (extra < 0)
	    {
	      printf ("extra should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'd':
	  delimiter = optarg[0];
	  break;
	case 'q':
	  nbr_quad = atoi (optarg);
	  if (nbr_quad < 2)
	    {
	      printf ("nbr_quad should be a integer bigger than 1\n");
	      return 0;
	    }
	  break;
	case 'm':
	  model = atoi (optarg);
	  if (model < 1 || (model > 3))
	    {
	      printf ("model should be 1, 2 or 3.\n");
	      return 0;
	    }
	  break;
	case 'e':
	  max_em_iter = atoi (optarg);
	  if (max_em_iter < 0)
	    {
	      printf ("max_em_iter should be a non-negative integer.\n");
	      return 0;
	    }
	  break;
	case 'n':
	  max_nr_iter = atoi (optarg);
	  if (max_nr_iter < 0)
	    {
	      printf ("max_nr_iter should be a non-negative integer.\n");
	      return 0;
	    }
	  break;
	case 'p':
	  response_functions = 1;
	  break;
	case 'G':
	  grouping = 1;
	  break;
	case 'V':
	  verbose++;
	  break;
	case 1:
	  slope_mean = atof (optarg);
	  if (slope_mean <= 0)
	    {
	      printf ("slope_mean should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 2:
	  slope_dev = atof (optarg);
	  if (slope_dev <= 0)
	    {
	      printf ("slope_dev should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 3:
	  thresh_mean = atof (optarg);
	  break;
	case 4:
	  thresh_dev = atof (optarg);
	  if (thresh_dev <= 0)
	    {
	      printf ("thresh_dev should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 5:
	  asymp_mean = atof (optarg);
	  if (asymp_mean <= 0 || asymp_mean >= 1)
	    {
	      printf
		("asymp_mean should be a real number between 0 and 1 excluded.\n");
	      return 0;
	    }
	  break;
	case 6:
	  asymp_weight = atof (optarg);
	  if (asymp_weight <= 0)
	    {
	      printf ("asymp_weight should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 'N':
	  slope_prior = 0;
	  thresh_prior = 0;
	  asymp_prior = 0;
	  break;
	case 7:
	  slope_prior = 0;
	  break;
	case 8:
	  thresh_prior = 1;
	  break;
	case 9:
	  asymp_prior = 0;
	  break;
	case 'A':
	  ability = 1;
	  break;
	case 10:
	  ability_prior = 0;
	  break;
	case 11:
	  quad_from = atof (optarg);
	  break;
	case 12:
	  quad_to = atof (optarg);
	  break;
	case 13:
	  true_string = optarg;
	  break;
	case 14:
	  false_string = optarg;
	  break;
	case 15:
	  blank_string = optarg;
	  break;
	case 'T':
	  trimming = 1;
	  break;
	case 'P':
	  penalized = 1;
	  break;
	case 'K':
	  kernel = 1;
	  break;
	case 's':
	  smooth_factor = atof (optarg);
	  if (smooth_factor <= 0)
	    {
	      printf ("smooth_factor should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 16:
	  slope_init = atof (optarg);
	  if (slope_init <= 0)
	    {
	      printf ("slope_init should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 17:
	  thresh_init = atof (optarg);
	  break;
	case 18:
	  asymp_init = atof (optarg);
	  if (asymp_init <= 0 || asymp_init >= 1)
	    {
	      printf
		("asymp_init should be a real number between 0 an 1 excluded.\n");
	      return 0;
	    }
	  break;
	case 19:
	  precision = atof (optarg);
	  if (precision <= 0)
	    {
	      printf ("precision should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case '?':
	  return 0;
	  break;
	default:
	  return 0;
	}
    }

  /* the non optional argument */
  if (optind < argc)
    {
      file_name = argv[optind];
    }
  else
    {
      printf ("Missing file_name.\n");
      printf ("Use -h for help.\n");
      return 0;
    }

  /* if 2PLM then disable the prior on the asymptote
     and set the asymptote to zero */
  if (model < 3)
    {
      asymp_prior = 0;
      asymp_init = 0.0;
    }

  /* if Rasch then disable the prior on the slope */
  if (model < 2)
    {
      slope_prior = 0;
    }

  /* set the library variable libirt_verbose */
  libirt_verbose = verbose;

  /* if using kernel or PMMLE enable the display of response functions */
  if (penalized || kernel)
    {
      response_functions = 1;
    }

  /* if using PMMLE check if nbr_quad is a power of 2 */
  if (penalized)
    {
      if (fabs (log (nbr_quad) / log (2.0)
		- (int) (log (nbr_quad) / log (2.0) + 0.5)) > 0.001)
	{
	  printf
	    ("nbr_quad should be a power of 2 when using penalized. Use 16, 32, 64, ... \n");
	  return 0;
	}
    }

  /* if using kernel or PMMLE the abilities can only be estimated by EAP */
  if (penalized || kernel)
    {
      if (!ability_prior)
	{
	  printf
	    ("Cannot use ML on the ability when using penalized or kernel, use EAP instead.\n");
	  return 0;
	}
    }

  return 1;
}

/** \brief The main function of the program. */
int
main (int argc, char *argv[])
{
  int i, k, j, nbr_pattern, ret_val, nbr_deg, nbr_notconverge;
  FILE *file;
  gsl_matrix_int *patterns=NULL, *patterns_group=NULL;
  gsl_vector *quad_points=NULL, *quad_weights=NULL,
    *slopes=NULL, *thresholds=NULL, *asymptotes=NULL,
    *slopes_stddev=NULL, *thresh_stddev=NULL, *asymp_stddev=NULL,
    *counts=NULL, *abilities=NULL, *abilities_stddev=NULL;
  gsl_vector_int *index=NULL;
  gsl_matrix *probs=NULL, *probs_stddev=NULL, *post;
  gsl_vector_int *ignore=NULL;
  gsl_vector_int *notconverge=NULL;

  /* parse the options and arguments */
  if (!parse_arg (argc, argv))
    {
      return 1;
    }

  /* printing the libirt version */
  if (verbose > 0)
    {
      printf ("Using version %s of libirt.\n", libirt_version);
    }

  /* open the file */
  if (NULL == (file = fopen (file_name, "r")))
    {
      printf ("Unable to open %s.\n", file_name);
      return 1;
    }

  /* read it */
  if (!read_bin_patterns_delimited (file, nbr_item, nbr_subject,
				    skip, delimiter, offset, extra,
				    true_string, false_string,
				    blank_string, trimming, &patterns))
    return 1;

  /* close it */
  fclose (file);

  nbr_subject = patterns->size1;
  nbr_item = patterns->size2;

  /* print the numbers of patterns and items read */
  if (verbose > 0)
    {
      printf ("%d subjects and %d items read.\n", nbr_subject, nbr_item);
    }

  /* if using PMMLE set the default smoothing factor */
  if (penalized && !smooth_factor)
    {
      smooth_factor = 4 * pow(nbr_subject, 0.2);
    }

  /* if using kernel set the default bandwith */
  if (kernel && !smooth_factor)
    {
      smooth_factor = 2.7 * pow(nbr_subject, -0.2);
    }

  /* group the identicals patterns together */
  if (grouping)
    {
      patterns_counts (patterns, &index, &patterns_group, &counts);
      nbr_pattern = patterns_group->size1;
      if (verbose > 0)
	{
	  printf ("%d patterns after grouping.\n", nbr_pattern);
	}
    }
  else
    {
      if (verbose > 0 && nbr_subject / nbr_item > 99)
	{
	  printf ("Grouping the patterns might speed up the process.\n");
	}
      patterns_group = patterns;
      nbr_pattern = patterns_group->size1;
      counts = NULL;
    }

  /* generate the quadrature points and weights */
  quad_points = gsl_vector_alloc(nbr_quad);
  quad_weights = gsl_vector_alloc(nbr_quad);
  quadrature (nbr_quad, quad_from, quad_to, quad_points, quad_weights);

  if (verbose > 0)
    printf ("Using %d quadrature points from %.5lg to %.5lg.\n",
	    nbr_quad, quad_from, quad_to);

  /* print the quadrature points and weights */
  if (verbose > 1)
    {
      printf ("The quadrature points and weights :\n");
      printf ("%10s  %10s  %10s\n", "NUM", "POINT", "WEIGHT");
      for (k = 0; k < nbr_quad; k++)
	{
	  printf ("%10d  %10.4lf  %10.4lf\n", k + 1,
		  gsl_vector_get (quad_points, k),
		  gsl_vector_get (quad_weights, k));
	}
      printf ("\n");
    }

  /* allocate the memory for the parameters
     and set the initials values of the parameters */
  if ((ability && ability_prior) || response_functions)
    {
      probs = gsl_matrix_alloc (nbr_item, nbr_quad);
      probs_stddev = gsl_matrix_alloc (nbr_item, nbr_quad);
      gsl_matrix_set_all (probs_stddev, 0.0);
    }

  if (penalized)
    {
      /* estimate the response functions by kernel */
      nadaraya_watson (2.7 * pow(nbr_subject, -0.2),
		       patterns, quad_points, quad_weights, probs, NULL);
    }
  else if(!kernel)
    {
      slopes = gsl_vector_alloc (nbr_item);
      thresholds = gsl_vector_alloc (nbr_item);
      asymptotes = gsl_vector_alloc (nbr_item);

      slopes_stddev = gsl_vector_alloc (nbr_item);
      thresh_stddev = gsl_vector_alloc (nbr_item);
      asymp_stddev = gsl_vector_alloc (nbr_item);

      gsl_vector_set_all (slopes, slope_init);
      gsl_vector_set_all (thresholds, thresh_init);
      gsl_vector_set_all (asymptotes, asymp_init);

      gsl_vector_set_all (slopes_stddev, 0.0);
      gsl_vector_set_all (thresh_stddev, 0.0);
      gsl_vector_set_all (asymp_stddev, 0.0);

      /* print the initials values of the parameters */
      if (verbose > 2)
	{
	  printf ("The initial parameters :\n");
	  printf ("%10s  %10s  %10s  %10s\n", "ITEM", "SLOPE", "THRESH",
		  "ASYMP");
	  for (i = 0; i < nbr_item; i++)
	    {
	      printf ("%10d  %10.5lf  %10.5lf  %10.5lf\n", i + 1,
		      gsl_vector_get (slopes, i),
		      gsl_vector_get (thresholds, i),
		      gsl_vector_get (asymptotes, i));
	    }
	  printf ("\n");
	}
    }

  /* check for degenerate items */
  ignore = gsl_vector_int_alloc(nbr_item);
  nbr_deg = set_ignore(patterns, probs, thresholds, ignore);
  if (verbose > 0 && nbr_deg > 0)
    {
      printf ("There is %d degenerate items.\n", nbr_deg);
    }

  /* reset the convergence flag */
  /* it will be the number of items that didn't converged */
  ret_val = 0;
  notconverge = gsl_vector_int_alloc(nbr_item);

  /* print the description of the estimation method */
  if (verbose > 0)
    {
      if(!kernel)
	{
	  printf ("EM iterations : %d.\n", max_em_iter);
	  printf ("Newton iterations : %d.\n", max_nr_iter);
	  printf ("Precision : %.5lg.\n", precision);
	}
      if (penalized)
	printf ("Using PMMLE with a smooth factor of %.5lg.\n",
		smooth_factor);
      else if (kernel)
	printf ("Using kernel regression with a bandwidth of %.5lg.\n",
		smooth_factor);
      else
	{
	  if (model == 1)
	    printf ("Using the Rasch model with a fixed slope of %.5lg.\n",
		    slope_init);
	  else if (model == 2)
	    printf ("Using the 2PLM.\n");
	  else if (model == 3)
	    printf ("Using the 3PLM.\n");
	  else
	    printf ("This model is not implemented yet !\n");

	  if (slope_prior)
	    printf
	      ("With a lognormal prior on the slopes (mu=%.5lg, sigma=%.5lg).\n",
	       slope_mean, slope_dev);

	  if (thresh_prior)
	    printf
	      ("With a normal prior on the thresholds (mu=%.5lg, sigma=%.5lg).\n",
	       thresh_mean, thresh_dev);

	  if (model > 2 && asymp_prior)
	    printf
	      ("With a beta prior on the asymptotes (p=%.5lg, m=%.5lg).\n",
	       asymp_mean, asymp_weight);
	}
      fflush (stdout);
    }

  if (penalized)
    /* estimate the response functions */
    ret_val =
      em_mple_wave (max_em_iter, max_nr_iter, precision, smooth_factor,
		    patterns_group, counts, quad_points, quad_weights,
		    probs, probs_stddev, ignore, &nbr_notconverge, notconverge, 0);
  else if (kernel)
    /* estimate the response functions */
    nadaraya_watson (smooth_factor, patterns,
		     quad_points, quad_weights,
		     probs, probs_stddev);
  else
    {
      /* estimate the parameters */
      ret_val = em_bme_3plm (model, max_em_iter, max_nr_iter, precision,
			     patterns_group, counts, quad_points,
			     quad_weights, slope_prior, slope_mean, slope_dev,
			     thresh_prior, thresh_mean, thresh_dev,
			     asymp_prior, asymp_mean, asymp_weight, slopes,
			     thresholds, asymptotes, slopes_stddev,
			     thresh_stddev, asymp_stddev, ignore, 
			     &nbr_notconverge, notconverge, 0);

      if (response_functions)
	{
	  /* compute the response functions */
	  probs_3plm (slopes, thresholds, asymptotes, quad_points, probs);
	}
    }

  /* print a message if some items didn't converged */
  if (verbose > 0 && nbr_notconverge > 0)
    {
      printf ("Warning : %d items didn't converged.\n", nbr_notconverge);
      printf ("Run with -VVVV to see the details.\n");
    }

  if (response_functions)
    {
      /* print the response functions */
      printf ("%10s", "POINT");
      for (i = 0; i < nbr_item; i++)
	printf ("   ITEM_%04d     SE_%04d", i + 1, i + 1);
      printf ("\n");
      for (k = 0; k < nbr_quad; k++)
	{
	  printf ("%10.5lf", gsl_vector_get (quad_points, k));
	  for (i = 0; i < nbr_item; i++)
	    printf ("  %10.5lf  %10.3lg",
		    gsl_matrix_get (probs, i, k),
		    gsl_matrix_get (probs_stddev, i, k));
	  printf ("\n");
	}
    }
  else
    {
      /* print the items parameters */
      printf ("%10s  %10s  %10s  %10s  %10s  %10s  %10s\n",
	      "ITEM", "SLOPE", "SLO_SE", "THRESH", "THR_SE", "ASYMP",
	      "ASY_SE");
      for (i = 0; i < nbr_item; i++)
	{
	  printf
	    ("%10d  %10.5lf  %10.3lg  %10.5lf  %10.3lg  %10.5lf  %10.3lg\n",
	     i + 1, gsl_vector_get (slopes, i), gsl_vector_get (slopes_stddev,
								i),
	     gsl_vector_get (thresholds, i), gsl_vector_get (thresh_stddev,
							     i),
	     gsl_vector_get (asymptotes, i), gsl_vector_get (asymp_stddev,
							     i));
	}
    }

  /* estimate the abilities */
  if (ability)
    {
      printf ("\n");

      /* allocate the memory for the abilities */
      abilities = gsl_vector_alloc (nbr_pattern);
      abilities_stddev = gsl_vector_alloc (nbr_pattern);
      post = gsl_matrix_alloc (nbr_pattern, nbr_quad);

      if (ability_prior || penalized || kernel)
	{
	  /* estimate the abilities by the bayes expected a posteriori method */
	  if (verbose > 0)
	    {
	      printf ("Estimating the abilities with prior (EAP).\n");
	      fflush (stdout);
	    }

	  if (!response_functions)
	    /* compute the response functions */
	    probs_3plm (slopes, thresholds, asymptotes, quad_points, probs);

	  posteriors (patterns_group, probs, quad_weights, post);

	  eap_abilities (post,
			 quad_points, quad_weights, abilities,
			 abilities_stddev);
	}
      else
	{
	  /* set the initial values for the abilities */
	  gsl_vector_set_all (abilities, 0.0);

	  /* reset the convergence flag */
	  /* it will be the number of patterns that didn't converged */
	  nbr_notconverge = 0;

	  /* estimate the abilities by the maximum likelihood method */
	  if (verbose > 0)
	    {
	      printf ("Estimating the abilities without prior (ML).\n");
	      fflush (stdout);
	    }
	  ret_val =
	    mle_abilities_3plm (max_nr_iter, precision, patterns_group,
				slopes, thresholds, asymptotes, abilities,
				abilities_stddev);

	  /* print a message if some patterns didn't converged */
	  if (verbose > 0 && ret_val > 0)
	    {
	      printf ("Warning : %d patterns didn't converged.\n", ret_val);
	      printf ("Run with -VVVV and without -G to see the details.\n");
	    }
	}

      /* print the abilities */
      printf ("%10s  %10s  %10s\n", "SUBJECT", "ABILITY", "ABI_SE");
      for (j = 0; j < nbr_subject; j++)
	{
	  if (grouping)
	    k = gsl_vector_int_get (index, j);
	  else
	    k = j;
	  printf ("%10d  %10.5lf  %10.3lg\n", j + 1,
		  gsl_vector_get (abilities, k),
		  gsl_vector_get (abilities_stddev, k));
	}
    }

  /* free the memory */
  gsl_matrix_int_free (patterns);
  gsl_vector_int_free (ignore);
  gsl_vector_int_free (notconverge);
  gsl_vector_free (quad_points);
  gsl_vector_free (quad_weights);
  if ((ability && ability_prior) || response_functions)
    {
      gsl_matrix_free (probs);
      gsl_matrix_free (probs_stddev);
    }
  if(!penalized && !kernel)
    {
      gsl_vector_free (slopes);
      gsl_vector_free (thresholds);
      gsl_vector_free (asymptotes);
      gsl_vector_free (slopes_stddev);
      gsl_vector_free (thresh_stddev);
      gsl_vector_free (asymp_stddev);
    }
  if (grouping)
    {
      gsl_matrix_int_free (patterns_group);
      gsl_vector_int_free (index);
      gsl_vector_free (counts);
    }
  if (ability)
    {
      gsl_matrix_free (post);
      gsl_vector_free (abilities);
      gsl_vector_free (abilities_stddev);
    }

  return 0;
}
